<?php

namespace AFEB\Widgets\FormBuilder;

use Elementor\Icons_Manager;
use Elementor\Utils;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * "Vertex Addons for Elementor" FormBuilder Widget Helper Class
 * 
 * @class Helper
 * @version 1.4.0
 */
class Helper
{
    public function form_fields_render_attributes($context, $i, $settings, $item)
    {
        $context->add_render_attribute(
            [
                'field-group' . $i => [
                    'class' => [
                        'afeb-form-field-type-' . $item['field_type'],
                        'afeb-field-group',
                        'afeb-column',
                        'afeb-field-group-' . esc_attr($item['field_id']),
                    ],
                ],
                'input' . $i => [
                    'type' => ('acceptance' === $item['field_type']) ? 'checkbox' : (($item['field_type'] === 'upload') ? 'file' :  $item['field_type']),
                    'name' => $this->get_attribute_name($item),
                    'id' => $this->get_attribute_id($item),
                    'class' => [
                        'afeb-form-field',
                        empty($item['css_classes']) ? '' : esc_attr($item['css_classes']),
                    ],
                ],
                'label' . $i => [
                    'for' => $this->get_attribute_id($item),
                    'class' => 'afeb-form-field-label',
                ],
            ]
        );

        if (empty($item['width']))
            $item['width'] = '100';

        $context->add_render_attribute('field-group' . $i, 'class', 'elementor-repeater-item-' . esc_attr($item['_id']));

        if ($settings['show_placeholders'] == 'true' && !Utils::is_empty($item['placeholder']))
            $context->add_render_attribute('input' . $i, 'placeholder', $item['placeholder']);

        if (!empty($item['field_value']))
            $context->add_render_attribute('input' . $i, 'value', $item['field_value']);

        if (!$settings['show_labels'])
            $context->add_render_attribute('label' . $i, 'class', 'afeb-hidden-element');

        if (!empty($item['required'])) {
            $class = 'afeb-form-field-required';
            if (!empty($settings['mark_required']))
                $class .= ' afeb-mark-required';

            $context->add_render_attribute('field-group' . $i, 'class', $class);
            $this->add_required_attribute($context, 'input' . $i);
        }
    }

    public function get_attribute_name($item)
    {
        return "form_fields[{$item['field_id']}]";
    }

    public function get_attribute_id($item)
    {
        $id_suffix = !empty($item['field_id']) ? $item['field_id'] : $item['_id'];
        return 'form-field-' . $id_suffix;
    }

    private function add_required_attribute($context, $element)
    {
        $context->add_render_attribute($element, 'required', 'required');
        $context->add_render_attribute($element, 'aria-required', 'true');
    }

    public function make_textarea_field($context, $item, $item_index)
    {
        $context->add_render_attribute('textarea' . $item_index, [
            'class' => [
                'afeb-form-field-textual',
                'afeb-form-field',
                'elementor-field-textual',
                esc_attr($item['css_classes'])
            ],
            'name' => $this->get_attribute_name($item),
            'id' => $this->get_attribute_id($item),
            'rows' => $item['rows'],
        ]);

        if ($context->get_settings_for_display()['show_placeholders'] == 'true' && !Utils::is_empty($item['placeholder'])) {
            $context->add_render_attribute('textarea' . $item_index, 'placeholder', $item['placeholder']);
        }

        if ($item['required']) {
            $this->add_required_attribute($context, 'textarea' . $item_index);
        }

        $value = empty($item['field_value']) ? '' : $item['field_value'];

        return '<textarea ' . $context->get_render_attribute_string('textarea' . $item_index) . '>' . $value . '</textarea>';
    }

    public function make_select_field($context, $item)
    {
        ob_start();
?>
        <div class="afeb-form-field afeb-select-wrap <?php echo esc_attr($item['css_classes']); ?>">
            <select
                id="<?php echo $this->get_attribute_id($item); ?>"
                class="afeb-form-field-textual elementor-field-textual"
                name="<?php echo $this->get_attribute_name($item); ?>"
                <?php echo $item['required'] ? 'required' : '' ?>>

                <?php
                $options_temp = [];
                $options = preg_split("/\\r\\n|\\r|\\n/", $item['field_options']);

                foreach ($options as $key => $option) {
                    $option_value = esc_attr($option);
                    $option_label = esc_html($option);

                    if (strpos($option, '|') !== false) {
                        list($label, $value) = explode('|', $option);
                        $option_value = esc_attr($value);
                        $option_label = esc_html($label);
                    }

                    if (isset($options_temp[$option_value]) || !trim($option_value)) {
                        continue;
                    }
                ?>
                    <option value="<?php echo esc_attr($option_value); ?>">
                        <?php
                        // PHPCS - $option_label is already escaped
                        echo $option_label; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped 
                        ?>
                    </option>
                <?php
                    $options_temp[$option_value] = $option_label;
                } ?>
            </select>
        </div>
    <?php
        $select = ob_get_clean();
        return $select;
    }

    public function make_radio_checkbox_field($context, $item, $type)
    {
        $options = preg_split("/\\r\\n|\\r|\\n/", $item['field_options']);
        $explode_check = explode('|', $options[0]);

        if (!isset($explode_check[1]) || (isset($explode_check[1]) && !trim($explode_check[1]))) {
            return '';
        }

        $options_temp = [];
        ob_start();
    ?>
        <div class="afeb-field-sub-group <?php echo esc_attr($item['css_classes']) . ' ' . esc_attr($item['inline_list']); ?>">
            <?php
            foreach ($options as $key => $option):
                $option_label = $option;
                $option_value = $option;

                if (strpos($option, '|') !== false) {
                    list($option_label, $option_value) = explode('|', $option);
                }

                if (isset($options_temp[$option_value]) || !trim($option_value)) {
                    continue;
                }
            ?>
                <span class="afeb-form-field-option" data-key="form-field-<?php echo esc_attr($item['field_id']); ?>">
                    <input
                        id="<?php echo esc_attr($this->get_attribute_id($item) . '-' . $key) ?>"
                        type="<?php echo esc_attr($type) ?>"
                        name="<?php echo esc_attr($this->get_attribute_name($item) . (($type == 'checkbox' && count($options) > 1) ? '[]' : '')); ?>"
                        value="<?php echo esc_attr($option_value); ?>"
                        <?php echo $item['required'] ? 'required' : '' ?>>
                    <?php if ($type == 'checkbox'): ?>
                        <input
                            type="hidden"
                            name="<?php echo esc_attr($this->get_attribute_name($item) . (($type == 'checkbox' && count($options) > 1) ? '[]' : '')); ?>">
                    <?php endif; ?>
                    <label for="<?php echo esc_attr($this->get_attribute_id($item) . '-' . $key) ?>"><?php echo esc_html($option_label); ?></label>
                </span>
            <?php
                $options_temp[$option_value] = $option_label;
            endforeach;
            ?>
        </div>
    <?php
        $radio_checkbox = ob_get_clean();
        return $radio_checkbox;
    }

    private function render_form_icon($context, $settings)
    {
    ?>
        <span <?php echo $context->get_render_attribute_string('icon-align'); ?>>
            <?php Icons_Manager::render_icon($settings['selected_button_icon'], ['aria-hidden' => 'true']); ?>
            <?php if (empty($settings['button_text'])): ?>
                <span class="afeb-hidden-element">
                    <?php echo esc_html__('Submit', 'addons-for-elementor-builder'); ?>
                </span>
            <?php endif; ?>
        </span>
    <?php
    }

    public function render_submit_button($context, $settings)
    {
    ?>
        <button type="submit" <?php echo $context->get_render_attribute_string('button'); ?>>
            <span <?php echo $context->get_render_attribute_string('content-wrapper'); ?>>
                <?php if (
                    !empty($settings['selected_button_icon']) &&
                    $settings['button_icon_align'] === 'left'
                ): ?>
                    <?php $this->render_form_icon($context, $settings); ?>
                <?php endif; ?>
                <?php if (!empty($settings['button_text'])): ?>
                    <span><?php echo $context->print_unescaped_setting('button_text'); ?></span>
                <?php endif; ?>
                <?php if (
                    !empty($settings['selected_button_icon']) &&
                    $settings['button_icon_align'] === 'right'
                ): ?>
                    <?php $this->render_form_icon($context, $settings); ?>
                <?php endif; ?>
            </span>
        </button>
<?php
    }
}
