<?php

namespace AFEB\Handler\Widgets;

use AFEB\Handler;
use AFEB\PostTypes\Submissions;
use AFEB\PRO\Helper as AFEBPHelper;
use AFEB\Widgets;

if (!defined('ABSPATH'))
{
    exit;
}

/**
 * "Vertex Addons for Elementor" FormBuilderHandler Class
 *
 * @class FormBuilderHandler
 * @version 1.4.0
 */
class FormBuilderHandler extends Handler
{
    /**
     * @var bool
     */
    private $ajax;

    /**
     * @var int
     */
    private $page_id;

    /**
     * @var int
     */
    private $form_id;

    /**
     * @var string
     */
    private $err_msg;

    /**
     * Initialize "Vertex Addons for Elementor" FormBuilderHandler
     *
     * @since 1.4.0
     */
    public function init()
    {
        $this->actions();
    }

    /**
     * FormBuilderHandler Class Actions
     *
     * @since 1.4.0
     */
    public function actions()
    {
        add_action('init', [$this, 'form_builder_handler']);
    }

    /**
     * Login or register user handler
     *
     * @since 1.4.0
     */
    public function form_builder_handler()
    {
        do_action('afeb/form_builder/before_form_handler', $_POST);

        if (isset($_POST['afeb_form_submit']))
        {
            $this->form_submit();
        }


        do_action('afeb/form_builder/after_form_handler', $_POST);
    }

    /**
     * Checks if there is an initial error
     *
     * @since 1.4.0
     */
    public function check_common_errors()
    {
        $this->ajax = wp_doing_ajax();
        $this->page_id = 0;
        $this->err_msg = '';

        if (!empty($_POST['page_id']))
        {
            $this->page_id = intval($_POST['page_id']);

            if (in_array(get_post_status($this->page_id), ['future', 'draft', 'pending']))
            {
                $this->err_msg = esc_html__('Please publish the page first and then try again', 'addons-for-elementor-builder');
            }
        }
        else
        {
            $this->err_msg = esc_html__('The page ID is not set', 'addons-for-elementor-builder');
        }

        $this->form_id = false;
        if (empty($this->err_msg))
        {
            if (!empty($_POST['form_id']))
            {
                $this->form_id = sanitize_text_field($_POST['form_id']);
                setcookie('afeb_form_submit_error_' . $this->form_id, '', time() + 2);
            }
            else
            {
                $this->err_msg = esc_html__('The form ID is not set', 'addons-for-elementor-builder');
            }
        }
        if (empty($this->err_msg))
        {
            if (empty($_POST['afeb_form_submit_nonce']))
                $this->err_msg = esc_html__('The submitted form is not secure, Nonce is not set', 'addons-for-elementor-builder');
        }
        if (empty($this->err_msg) && !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['afeb_form_submit_nonce'])), 'afeb_form_submit_action'))
            $this->err_msg = esc_html__('Security token did not match', 'addons-for-elementor-builder');

        if (!empty($this->err_msg))
        {
            if ($this->ajax)
            {
                wp_send_json_error($this->err_msg);
            }

            if (!empty($this->form_id))
            {
                setcookie('afeb_form_submit_error_' . $this->form_id, esc_html($this->err_msg), time() + 2);
            }

            if (!empty($_SERVER['HTTP_REFERER']))
            {
                wp_safe_redirect(esc_url_raw($_SERVER['HTTP_REFERER']));
                exit();
            }
        }
    }

    /**
     * Form Submit
     *
     * @since 1.4.0
     */
    public function form_submit()
    {
        $this->check_common_errors();

        $settings = Widgets::get_widget_preview_settings($this->page_id, $this->form_id);
        $form_fields = !empty($_POST['form_fields']) && is_array($_POST['form_fields']) ? array_values($_POST['form_fields']) : [];
        $settings_form_fields = !empty($settings['form_fields']) ? $settings['form_fields'] : [];
        $normal_settings_form_fields = $settings_form_fields;
        $has_recaptcha = false;

        foreach ($normal_settings_form_fields as $form_field_index => $form_field)
        {
            if (!empty($form_field['field_type']))
            {
                if (
                    $form_field['field_type'] == 'step' ||
                    $form_field['field_type'] == 'html'
                )
                {
                    unset($normal_settings_form_fields[$form_field_index]);
                }

                if ($form_field['field_type'] == 'recaptcha_v3')
                {
                    $has_recaptcha = true;
                    unset($normal_settings_form_fields[$form_field_index]);
                }
            }
        }

        $normal_settings_form_fields = array_values($normal_settings_form_fields);

        $new_form_fields = [];
        $loop_break = false;

        foreach ($normal_settings_form_fields as $form_field_key => $form_field)
        {
            $new_form_fields[$form_field_key]['_id'] = !empty($form_field['field_id']) ? sanitize_text_field($form_field['field_id']) : '';
            $new_form_fields[$form_field_key]['label'] = !empty($form_field['field_label']) ? sanitize_text_field($form_field['field_label']) : '';
            $new_form_fields[$form_field_key]['type'] = !empty($form_field['field_type']) ? sanitize_text_field($form_field['field_type']) : '';
            $new_form_fields[$form_field_key]['options'] = !empty($form_field['field_options']) ? sanitize_text_field($form_field['field_options']) : '';

            // Check for reCAPTCHA
            $ajax_verify_recaptcha = get_option('_afebp_ajax_verify_recaptcha');
            $ajax_verify_recaptcha = $ajax_verify_recaptcha ?: 'no';
            if (
                !$this->ajax &&
                $has_recaptcha &&
                $ajax_verify_recaptcha == 'no'
            )
            { //Check PRO is Active
                update_option('_afebp_ajax_verify_recaptcha', 'no');

                $recaptcha_response = $_POST['afeb_form_field_recaptcha_v3'] ?? '';
                $is_valid_recaptcha = false;

                if (class_exists('\AFEB\PRO\Helper'))
                {
                    $is_valid_recaptcha = AFEBPHelper::verify_recaptcha($recaptcha_response);
                }

                if ($is_valid_recaptcha !== true)
                {
                    $loop_break = true;
                    break;
                }
            }

            switch ($new_form_fields[$form_field_key]['type'])
            {
                case 'text':
                case 'radio':
                case 'select':
                case 'search':
                case 'date':
                case 'time':
                case 'password':
                case 'html':
                case 'hidden':
                    $new_form_fields[$form_field_key]['value'] = !empty($form_fields[$form_field_key]) ? sanitize_text_field($form_fields[$form_field_key]) : '';
                    break;
                case 'checkbox':
                    if (!empty($form_fields[$form_field_key]))
                    {
                        if (is_array($form_fields[$form_field_key]))
                        {
                            $new_form_fields[$form_field_key]['value'] = map_deep($form_fields[$form_field_key], 'sanitize_text_field');
                        }
                        else
                        {
                            $new_form_fields[$form_field_key]['value'] = sanitize_text_field($form_fields[$form_field_key]);
                        }
                    }
                    else
                    {
                        $new_form_fields[$form_field_key]['value'] = '';
                    }
                    break;
                case 'tel':
                case 'number':
                    $new_form_fields[$form_field_key]['value'] = !empty($form_fields[$form_field_key]) ? preg_replace('/[^\d+]/', '', $form_fields[$form_field_key]) : '';
                    break;
                case 'email':
                    $new_form_fields[$form_field_key]['value'] = !empty($form_fields[$form_field_key]) ? sanitize_email($form_fields[$form_field_key]) : '';
                    break;
                case 'url':
                    $new_form_fields[$form_field_key]['value'] = !empty($form_fields[$form_field_key]) ? sanitize_url($form_fields[$form_field_key]) : '';
                    break;
                case 'textarea':
                    $new_form_fields[$form_field_key]['value'] = !empty($form_fields[$form_field_key]) ? sanitize_textarea_field($form_fields[$form_field_key]) : '';
                    break;
                case 'upload':
                    $new_form_fields[$form_field_key]['value'] = '';

                    if (isset($form_fields[$form_field_key]))
                    {
                        $value = explode('|', rtrim($form_fields[$form_field_key], '|'));
                        $value = map_deep($value, 'sanitize_url');

                        $new_form_fields[$form_field_key]['value'] = $value;
                    }
                    break;
            }
        }

        // reCAPTCHA Error
        if ($loop_break)
        {
            $this->err_msg = esc_html__('Recaptcha Error', 'addons-for-elementor-builder');
            setcookie('afeb_form_submit_error_' . $this->form_id, esc_html($this->err_msg), time() + 2);

            if (!empty($_SERVER['HTTP_REFERER']))
            {
                wp_safe_redirect(esc_url_raw($_SERVER['HTTP_REFERER']));
                exit();
            }
        }

        update_option('_afebp_ajax_verify_recaptcha', 'no');

        $submit_actions = !empty($settings['submit_actions']) && is_array($settings['submit_actions']) ? $settings['submit_actions'] : [];
        $new_form_fields = array_values($new_form_fields);
        $redirect = in_array('redirect', $submit_actions) && !empty($settings['redirect_to']) ? sanitize_url($settings['redirect_to']) : '';
        $error = false;

        if (in_array('submission', $submit_actions))
        {
            $form_title = isset($_POST['form_name']) ? sanitize_text_field($_POST['form_name']) : '';
            $id = wp_insert_post([
                'post_status' => 'publish',
                'post_title' => $form_title,
                'post_type' => Submissions::SUBMISSIONS_POST_TYPE,
                'post_name' => Submissions::SUBMISSIONS_POST_TYPE,
            ]);

            if ($id)
            {
                if ($form_title == '')
                {
                    $id = intval($id);
                    wp_update_post([
                        'ID' => $id,
                        'post_title' => "Vertex Submission_$id",
                    ]);
                }

                $form_name = isset($_POST['form_name']) ? sanitize_text_field($_POST['form_name']) : '';

                if (trim($form_name))
                    update_post_meta($id, 'afeb_form_name', $form_name);

                update_post_meta($id, 'afeb_form_id', $this->form_id);
                update_post_meta($id, 'afeb_page_id', $this->page_id);
                update_post_meta($id, 'afeb_user_ip', $this->get_client_ip());
                update_post_meta($id, 'afeb_user_agent', sanitize_textarea_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])));
                update_post_meta($id, 'afeb_form_fields', $new_form_fields);
            }
            else
            {
                $error = true;
            }
        }

        if (in_array('email', $submit_actions))
        {
            if ($this->send_email($settings, $new_form_fields) === false)
            {
                $error = true;
            }
        }

        if ($error)
        {
            $this->err_msg = !empty($settings['error_message']) ?
                $settings['error_message'] : esc_html__('The operation was unsuccessful.', 'addons-for-elementor-builder');

            if ($this->ajax)
            {
                wp_send_json_error($this->err_msg);
            }

            setcookie('afeb_form_submit_error_' . $this->form_id, esc_html($this->err_msg), time() + 2);
        }
        else
        {
            $sucs_msg = !empty($settings['success_message']) ?
                $settings['success_message'] : esc_html__('The operation was successful.', 'addons-for-elementor-builder');

            if ($this->ajax)
            {
                wp_send_json_success($sucs_msg);
            }

            setcookie('afeb_form_submit_success_' . $this->form_id, esc_html($sucs_msg), time() + 2);
        }

        if ($redirect != '')
        {
            wp_safe_redirect(esc_url_raw($redirect));
            exit();
        }

        if (isset($_SERVER['HTTP_REFERER']))
        {
            wp_safe_redirect(esc_url_raw($_SERVER['HTTP_REFERER']));
            exit();
        }
    }

    public function _unstable_get_super_global_value($super_global, $key)
    {
        if (!isset($super_global[$key]))
            return null;

        if ($super_global === $_FILES)
        {
            $super_global[$key]['name'] = sanitize_file_name($super_global[$key]['name']);
            return $super_global[$key];
        }

        return wp_kses_post_deep(wp_unslash($super_global[$key]));
    }

    public function get_client_ip()
    {
        $server_ip_keys = [
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR',
        ];

        foreach ($server_ip_keys as $key)
        {
            $value = $this->_unstable_get_super_global_value($_SERVER, $key);
            if ($value && filter_var($value, FILTER_VALIDATE_IP))
                return $value;
        }

        // Fallback Local IP
        return '127.0.0.1';
    }

    public function send_email($settings = [], $form_fields = [])
    {
        $GLOBALS['afeb_form_fields'] = $form_fields;

        $content_type = isset($settings['email_content_type']) ? sanitize_text_field($settings['email_content_type']) : 'html';

        $line_break = $content_type == 'html' ? '<br>' : "\n";
        $email_content = isset($settings['email_content']) ? sanitize_textarea_field($settings['email_content']) : '[all-fields]';

        if ($email_content == '[all-fields]' || str_contains($email_content, '[all-fields]'))
        {
            $all_email_content = [];
            $replace_shortcode_with_value = function ($matches)
            {
                $field_id = $matches[1];

                foreach ($GLOBALS['afeb_form_fields'] as $form_field)
                {
                    $id = $form_field['_id'] ?? '';

                    if ($id == $field_id)
                    {
                        $value = $form_field['value'] ?? '';
                        return is_array($value) ? implode("\n", $value) . "\n" : $value . "\n";
                    }
                }

                return '';
            };

            foreach ($GLOBALS['afeb_form_fields'] as $form_field)
            {
                $label = $form_field['label'] ?? '';
                $value = $form_field['value'] ?? '';

                $all_email_content[] = is_array($value) ? trim($label) . ': ' . implode("\n", $value) . "\n" : trim($label) . ': ' . $value . "\n";
            }

            $all_email_content = implode("\n", $all_email_content);
            $processed_message = str_replace('[all-fields]', $all_email_content, $email_content);
            $processed_message = preg_replace_callback(
                '/\[id="([^"]+)"]/',
                $replace_shortcode_with_value,
                $processed_message
            );
        }
        else
        {
            $replace_shortcode_with_value = function ($matches)
            {
                $field_id = $matches[1];

                foreach ($GLOBALS['afeb_form_fields'] as $form_field)
                {
                    $id = $form_field['_id'] ?? '';

                    if ($id == $field_id)
                    {
                        $label = $form_field['label'] ?? '';
                        $value = $form_field['value'] ?? '';

                        return is_array($value) ? trim($label) . ': ' . implode("\n", $value) . "\n" : trim($label) . ': ' . $value . "\n";
                    }
                }

                return '';
            };

            $processed_message = preg_replace_callback(
                '/\[id="([^"]+)"]/',
                $replace_shortcode_with_value,
                $email_content
            );
        }

        $meta_keys = $settings['form_metadata'] ?? ['date', 'time', 'credit'];
        $meta_fields = [];

        foreach ($meta_keys as $metadata_type)
        {
            switch ($metadata_type)
            {
                case 'date':
                    $meta_fields['date'] = [
                        'title' => esc_html__('Date', 'addons-for-elementor-builder'),
                        'value' => date_i18n(get_option('date_format')),
                    ];
                    break;

                case 'time':
                    $meta_fields['time'] = [
                        'title' => esc_html__('Time', 'addons-for-elementor-builder'),
                        'value' => date_i18n(get_option('time_format')),
                    ];
                    break;

                case 'page_url':
                    $value = isset($_POST['referrer']) ? sanitize_url($_POST['referrer']) : '';
                    $meta_fields['page_url'] = [
                        'title' => esc_html__('Page URL', 'addons-for-elementor-builder'),
                        'value' => $value,
                    ];
                    break;

                case 'page_title':
                    $value = isset($_POST['referer_title']) ? sanitize_url($_POST['referer_title']) : '';
                    $meta_fields['page_title'] = [
                        'title' => esc_html__('Page Title', 'addons-for-elementor-builder'),
                        'value' => $value,
                    ];
                    break;

                case 'user_agent':
                    $meta_fields['user_agent'] = [
                        'title' => esc_html__('User Agent', 'addons-for-elementor-builder'),
                        'value' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_textarea_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
                    ];
                    break;

                case 'remote_ip':
                    $meta_fields['remote_ip'] = [
                        'title' => esc_html__('Remote IP', 'addons-for-elementor-builder'),
                        'value' => $this->get_client_ip(),
                    ];
                    break;

                case 'credit':
                    $meta_fields['credit'] = [
                        'title' => esc_html__('Powered by', 'addons-for-elementor-builder'),
                        'value' => esc_html__('Vertex Addons', 'addons-for-elementor-builder'),
                    ];
                    break;
            }
        }

        $email_meta = [];

        foreach ($meta_fields as $value)
            $email_meta[] = $value['title'] . ': ' . $value['value'];

        $fields = [
            'email_to' => isset($settings['email_to']) ?
                sanitize_email($settings['email_to']) : get_option('admin_email'),

            /* translators: %s: Site title. */
            'email_subject' => isset($settings['email_subject']) ?
                sanitize_text_field($settings['email_subject']) : sprintf(esc_html__('New message from "%s"', 'addons-for-elementor-builder'), get_bloginfo('name')),

            'email_content' => $processed_message . $line_break . '-----' . $line_break . implode($line_break, $email_meta),

            'email_from_name' => isset($settings['email_from_name']) ?
                sanitize_text_field($settings['email_from_name']) : get_bloginfo('name'),

            'email_from' => isset($settings['email_from']) ?
                sanitize_email($settings['email_from']) : get_bloginfo('admin_email'),

            'email_reply_to' => isset($settings['email_reply_to']) ?
                sanitize_email($settings['email_reply_to']) : '',

            'email_to_cc' => $settings['email_to_cc'] ?? '',

            'email_to_bcc' => $settings['email_to_bcc'] ?? '',
        ];

        $headers = sprintf('From: %s <%s>' . "\r\n", $fields['email_from_name'], $fields['email_from']);
        $headers .= sprintf('Reply-To: %s' . "\r\n", $fields['email_reply_to']);

        if ($content_type == 'html')
            $headers .= 'Content-Type: text/html; charset=UTF-8' . "\r\n";


        $cc_header = '';
        if (!empty($fields['email_to_cc']))
            $cc_header = 'Cc: ' . $fields['email_to_cc'] . "\r\n";

        $email_sent = wp_mail(
            $fields['email_to'],
            $fields['email_subject'],
            $fields['email_content'],
            $headers . $cc_header
        );

        if (!empty($fields['email_to_bcc']))
        {
            $bcc_emails = explode(',', $fields['email_to_bcc']);
            foreach ($bcc_emails as $bcc_email)
            {
                wp_mail(
                    trim($bcc_email),
                    $fields['email_subject'],
                    $fields['email_content'],
                    $headers
                );
            }
        }

        return (bool) $email_sent;
    }
}
