<?php

namespace AFEB\Controls;

use AFEB\Helper as AFEBHelper;
use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Css_Filter;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Text_Stroke;
use Elementor\Group_Control_Typography;
use Elementor\Utils;

/**
 * "Vertex Addons for Elementor" Controls Helper Class
 *
 * @version 1.0.0
 */
class Helper
{
    /**
     * @var Widget_Base
     */
    private $widget;
    private $responsive = false;

    public function __construct($widget)
    {
        $this->widget = $widget;
    }

    public function responsive(): Helper
    {
        $this->responsive = true;
        return $this;
    }

    public static function get($widget): Helper
    {
        return (new Helper($widget));
    }

    private function merge(array $default, array $args): array
    {
        return array_merge($default, $args);
    }

    private function add(string $id, array $args)
    {
        // Add Responsive Control
        if ($this->responsive) {
            $this->widget->add_responsive_control($id, $args);

            $this->responsive = false;
            return;
        }

        // Regular Control
        $this->widget->add_control($id, $args);
    }

    public function section(string $id, array $args, $callback = null)
    {
        $this->widget->start_controls_section($id, $args);

        $callback && $callback();

        $this->widget->end_controls_section();
    }

    public function tab_advanced_section(string $id, array $args, $callback = null)
    {
        $args['tab'] = Controls_Manager::TAB_ADVANCED;
        $this->section($id, $args, $callback);
    }

    public function tab_content_section(string $id, array $args, $callback = null)
    {
        $args['tab'] = Controls_Manager::TAB_CONTENT;
        $this->section($id, $args, $callback);
    }

    public function tab_settings_section(string $id, array $args, $callback = null)
    {
        $args['tab'] = Controls_Manager::TAB_SETTINGS;
        $this->section($id, $args, $callback);
    }

    public function tab_style_section(string $id, array $args, $callback = null)
    {
        $args['tab'] = Controls_Manager::TAB_STYLE;
        $this->section($id, $args, $callback);
    }

    public function tabs(string $id, array $tabs)
    {
        $this->widget->start_controls_tabs($id);

        foreach ($tabs as $tab_id => $tab) {
            $this->widget->start_controls_tab($tab_id, $tab);

            isset($tab['callback']) && is_callable($tab['callback']) && $tab['callback']();

            $this->widget->end_controls_tab();
        }

        $this->widget->end_controls_tabs();
    }

    public function control(string $id, array $args)
    {
        $this->add($id, $args);
    }

    public function select(string $id, array $args)
    {
        $default = [
            'type' => Controls_Manager::SELECT,
            'label_block' => false,
        ];

        if (!empty($args['meta_options'])) {
            $name = str_replace('afeb_', '', $args['meta_options'][0]);
            $default['options'] = AFEBHelper::fhook($args['meta_options'][1], "afeb/$name/controls/options/$id");
            unset($args['meta_options']);
        }

        $this->add($id, $this->merge($default, $args));
    }

    public function select2(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
        ], $args));
    }

    /**
     * Displays a dynamic select box control
     * 
     * @since 1.3.0
     * 
     * @param string $id
     * @param array $args
     */
    public function dynamic_select(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => 'afeb_dynamic_select',
            'label_block' => true,
        ], $args));
    }

    public function url(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'label' => esc_html__('Link', 'addons-for-elementor-builder'),
            'type' => Controls_Manager::URL,
            'default' => [
                'nofollow' => false,
                'is_external' => true,
            ],
            'dynamic' => [
                'active' => true,
            ],
            'placeholder' => esc_html__('https://your-link.com', 'addons-for-elementor-builder'),
            'label_block' => true,
        ], $args));
    }

    public function choose(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::CHOOSE,
            'toggle' => false,
        ], $args));
    }

    /**
     * Elementor popover toggle control
     * 
     * @since 1.5.0
     * 
     * @param string $id
     * @param array $args
     * @param object $callback
     */
    public function popover_toggle(string $id, array $args, $callback = null)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::POPOVER_TOGGLE,
        ], $args));

        $this->widget->start_popover();

        isset($callback) && is_callable($callback) && $callback();

        $this->widget->end_popover();
    }

    public function slider(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::SLIDER,
            'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
            'default' => [
                'unit' => 'px',
            ],
            'tablet_default' => [
                'unit' => 'px',
            ],
            'mobile_default' => [
                'unit' => 'px',
            ],
            'range' => [
                '%' => [
                    'min' => 1,
                    'max' => 100,
                ],
                'px' => [
                    'min' => 1,
                    'max' => 1000,
                ],
                'vw' => [
                    'min' => 1,
                    'max' => 100,
                ],
            ],
        ], $args));
    }

    public function dimensions(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        ], $args));
    }

    public function hover_animation(string $id, array $args = [])
    {
        $this->add($id, $this->merge([
            'label' => esc_html__('Hover Animation', 'addons-for-elementor-builder'),
            'type' => Controls_Manager::HOVER_ANIMATION,
        ], $args));
    }

    public function color(string $id, array $args = [])
    {
        $this->add($id, $this->merge([
            'label' => esc_html__('Color', 'addons-for-elementor-builder'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
        ], $args));
    }

    public function divider(string $id, array $args = [])
    {
        $this->widget->add_control($id, $this->merge([
            'type' => Controls_Manager::DIVIDER,
            'style' => 'thick',
        ], $args));
    }

    public function media(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'label' => esc_html__('Image', 'addons-for-elementor-builder'),
            'type' => Controls_Manager::MEDIA,
            'default' => ['url' => Utils::get_placeholder_image_src()],
        ], $args));
    }

    public function image_size(array $args)
    {
        $this->widget->add_group_control(Group_Control_Image_Size::get_type(), $this->merge([
            'default' => 'full',
            'exclude' => ['custom'],
            'separator' => '',
        ], $args));
    }

    public function background(array $args)
    {
        $this->widget->add_group_control(Group_Control_Background::get_type(), $this->merge([
            'name' => 'background',
            'selector' => '{{WRAPPER}}',
        ], $args));
    }

    public function border(array $args)
    {
        $this->widget->add_group_control(Group_Control_Border::get_type(), $this->merge([
            'name' => 'border',
            'selector' => '{{WRAPPER}}',
            'separator' => 'before',
        ], $args));
    }

    public function button(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::BUTTON,
            'label_block' => true,
            'show_label' => false,
        ], $args));
    }

    public function padding(string $id, array $args = [])
    {
        $args['label'] = esc_html__('Padding', 'addons-for-elementor-builder');
        $this->dimensions($id, $args);
    }

    public function margin(string $id, array $args = [])
    {
        $args['label'] = esc_html__('Margin', 'addons-for-elementor-builder');
        $this->dimensions($id, $args);
    }

    public function border_radius(string $id, array $args = [])
    {
        $args['label'] = esc_html__('Border Radius', 'addons-for-elementor-builder');
        $this->dimensions($id, $args);
    }

    public function css_filters(array $args)
    {
        $this->widget->add_group_control(Group_Control_Css_Filter::get_type(), $this->merge([
            'name' => 'css_filters',
            'selector' => '{{WRAPPER}} img',
        ], $args));
    }

    public function box_shadow(array $args)
    {
        $this->widget->add_group_control(Group_Control_Box_Shadow::get_type(), $this->merge([
            'name' => 'box_shadow',
            'selector' => '{{WRAPPER}}',
        ], $args));
    }

    public function text_shadow(array $args)
    {
        $this->widget->add_group_control(Group_Control_Text_Shadow::get_type(), $this->merge([
            'name' => 'text_shadow',
            'selector' => '{{WRAPPER}}',
        ], $args));
    }

    public function text_stroke(array $args)
    {
        $this->widget->add_group_control(Group_Control_Text_Stroke::get_type(), $this->merge([
            'name' => 'text_stroke',
            'selector' => '{{WRAPPER}}',
        ], $args));
    }

    public function typography(array $args)
    {
        $this->widget->add_group_control(Group_Control_Typography::get_type(), $this->merge([
            'name' => 'typography',
            'selector' => '{{WRAPPER}}',
            'global' => [
                'default' => Global_Typography::TYPOGRAPHY_TEXT,
            ],
        ], $args));
    }

    public function alignment(string $id, array $args = [])
    {
        $options = [
            'left' => ['title' => esc_html__('Left', 'addons-for-elementor-builder'), 'icon' => 'eicon-text-align-left',],
            'center' => ['title' => esc_html__('Center', 'addons-for-elementor-builder'), 'icon' => 'eicon-text-align-center',],
            'right' => ['title' => esc_html__('Right', 'addons-for-elementor-builder'), 'icon' => 'eicon-text-align-right',],
        ];

        if (!empty($args['justify'])) {
            $options['justify'] = [
                'title' => esc_html__('Justified', 'addons-for-elementor-builder'),
                'icon' => 'eicon-text-align-justify',
            ];
        }

        $this->choose($id, $this->merge([
            'label' => esc_html__('Alignment', 'addons-for-elementor-builder'),
            'options' => $options,
            'selectors' => [
                '{{WRAPPER}}' => 'text-align: {{VALUE}};',
            ],
        ], $args));
    }

    public function image_fit(string $id, array $args = [])
    {
        $this->select($id, $this->merge([
            'label' => esc_html__('Image Fit', 'addons-for-elementor-builder'),
            'options' => [
                '' => esc_html__('Default', 'addons-for-elementor-builder'),
                'fill' => esc_html__('Fill', 'addons-for-elementor-builder'),
                'cover' => esc_html__('Cover', 'addons-for-elementor-builder'),
                'contain' => esc_html__('Contain', 'addons-for-elementor-builder'),
                'scale-down' => esc_html__('Scale Down', 'addons-for-elementor-builder'),
            ],
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} img' => 'object-fit: {{VALUE}};',
            ],
        ], $args));
    }

    public function opacity(string $id, array $args = [])
    {
        $this->slider($id, $this->merge([
            'label' => esc_html__('Opacity', 'addons-for-elementor-builder'),
            'size_units' => ['px'],
            'default' => ['unit' => 'px'],
            'tablet_default' => ['unit' => 'px'],
            'mobile_default' => ['unit' => 'px'],
            'range' => [
                'px' => [
                    'max' => 1,
                    'min' => 0.10,
                    'step' => 0.01,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}}' => 'opacity: {{SIZE}};',
            ],
        ], $args));
    }

    public function duration(string $id, array $args = [])
    {
        $this->slider($id, $this->merge([
            'label' => esc_html__('Duration (s)', 'addons-for-elementor-builder'),
            'size_units' => ['px'],
            'default' => ['unit' => 'px'],
            'tablet_default' => ['unit' => 'px'],
            'mobile_default' => ['unit' => 'px'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 3,
                    'step' => 0.1,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}}' => 'transition-duration: {{SIZE}}s',
            ],
        ], $args));
    }

    public function text_color(string $id, array $args = [])
    {
        $this->color($id, $this->merge([
            'label' => esc_html__('Text Color', 'addons-for-elementor-builder'),
        ], $args));
    }

    public function background_color(string $id, array $args = [])
    {
        $this->color($id, $this->merge([
            'label' => esc_html__('Background Color', 'addons-for-elementor-builder'),
        ], $args));
    }

    /**
     * Displays an HTML content in the panel
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function raw_html(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::RAW_HTML,
        ], $args));
    }

    /**
     * Allows you to build repeatable blocks of fields
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function repeater(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::REPEATER,
            'label' => esc_html__('Items', 'addons-for-elementor-builder')
        ], $args));
    }

    /**
     * Elementor switcher control displays an on/off
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function switcher(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::SWITCHER,
        ], $args));
    }

    /**
     * Elementor switcher control displays an show/hide
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function sh_switcher(string $id, array $args)
    {
        $args['label_on'] = esc_html__('Show', 'addons-for-elementor-builder');
        $args['label_off'] = esc_html__('Hide', 'addons-for-elementor-builder');

        $this->Switcher($id, $args);
    }

    /**
     * Elementor switcher control displays an yes/no
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function yn_switcher(string $id, array $args)
    {
        $args['label_on'] = esc_html__('Yes', 'addons-for-elementor-builder');
        $args['label_off'] = esc_html__('No', 'addons-for-elementor-builder');

        $this->Switcher($id, $args);
    }

    /**
     * Displays a simple text input field
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function text(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::TEXT,
            'dynamic' => [
                'active' => true,
            ],
        ], $args));
    }

    /**
     * Displays a simple text area input field
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function text_area(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::TEXTAREA,
            'rows' => 3,
            'dynamic' => [
                'active' => true,
            ],
        ], $args));
    }

    /**
     * Displays a code editor input field
     * 
     * @since 1.5.0
     * 
     * @param string $id
     * @param array $args
     */
    public function code(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::CODE,
            'language' => 'css',
            'render_type' => 'ui',
        ], $args));
    }

    /**
     * Displays a simple WYSIWYG input field
     * 
     * @since 1.5.0
     * 
     * @param string $id
     * @param array $args
     */
    public function wysiwyg(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::WYSIWYG,
            'label' => esc_html__('Content', 'addons-for-elementor-builder'),
            'dynamic' => [
                'active' => true,
            ],
        ], $args));
    }

    /**
     * Displays a simple number input field
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function number(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::NUMBER,
            'min' => 0,
            'step' => 1,
        ], $args));
    }

    /**
     * Displays the icons chooser
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function icons(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'label' => esc_html__('Icon', 'addons-for-elementor-builder'),
            'type' => Controls_Manager::ICONS,
            'skin' => 'inline',
            'label_block' => false
        ], $args));
    }

    /**
     * Adds a hidden input field to the panel
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function hidden(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::HIDDEN
        ], $args));
    }

    /**
     * Displays a text heading between controls in the panel
     * 
     * @since 1.0.1
     * 
     * @param string $id
     * @param array $args
     */
    public function heading(string $id, array $args)
    {
        $this->add($id, $this->merge([
            'type' => Controls_Manager::HEADING
        ], $args));
    }
}
